//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2006-2011 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.IO;
using WeifenLuo.WinFormsUI;

namespace SID_monitor
{
    public partial class GOESDockablePanel : DockContent
    {

        public GOESDockablePanel()
        {
            InitializeComponent();

            // label GOES database Update
            updateLabelGOESUpdate();

        }


        #region Properties
        public bool PrimaryGOESChecked
        {
            get
            {
                return this.checkBoxPrimaryGOES.Checked;
            }
            set
            {
                this.checkBoxPrimaryGOES.Checked = value;
            }
        }

        public bool SecondaryGOESChecked
        {
            get
            {
                return this.checkBoxSecondaryGOES.Checked;
            }
            set
            {
                this.checkBoxSecondaryGOES.Checked = value;
            }
        }

        public string GOESLastUpdateText
        {
            get
            {
                return this.labelLastGOESUpdateValue.Text;
            }
            set
            {
                this.labelLastGOESUpdateValue.Text = value;
            }
        }

        #endregion

        #region Checkboxes Methods
        private void checkBoxPrimaryGOES_CheckedChanged(object sender, EventArgs e)
        {
            OnPrimaryGOESShowChanged(new GOESShowChangeEventArgs(checkBoxPrimaryGOES.Checked));
        }

        private void checkBoxSecondaryGOES_CheckedChanged(object sender, EventArgs e)
        {
            OnSecondaryGOESShowChanged(new GOESShowChangeEventArgs(checkBoxSecondaryGOES.Checked));
        }
        #endregion

        #region GOES Pane Methods
        /// <summary>
        /// An immediate refresh of the GOES database is asked
        /// </summary>
        private void buttonUpdateGOES_Click(object sender, EventArgs e)
        {
            OnGOESDownloadRequested(e);
        }

        /// <summary>
        /// Updates the label on the GOES pane that indicates the last GOES database update time
        /// </summary>
        public void updateLabelGOESUpdate()
        {
            if (!File.Exists(SID_monitor.Properties.Settings.Default.RRDToolGOES))
            {
                this.labelLastGOESUpdateValue.Text = "          N/A";
            }
            else
            {
                SIDMonitorDateTime lastGOESDatabaseUpdateTime;
                try
                {
                    lastGOESDatabaseUpdateTime = RRDTool.getLastDatabaseUpdateTime(SID_monitor.Properties.Settings.Default.RRDToolGOES, SID_monitor.Program.MainForm.outputTextBoxDockablePanel.Handle);
                }
                catch
                {
                    this.labelLastGOESUpdateValue.Text = "          N/A";
                    return;
                }
                this.labelLastGOESUpdateValue.Text = "          " + lastGOESDatabaseUpdateTime.WinTime.ToString("dd MMM yyyy HH:mm:ss") + " UTC";
            }
        }
        #endregion

        #region Events

        /// <summary>
        /// PrimaryGOESShowChanged, SecondaryGOESShowChanged Events
        /// </summary>

        public class GOESShowChangeEventArgs : EventArgs
        {
            private bool show = false;

            public GOESShowChangeEventArgs(bool show)
            {
                this.show = show;
            }

            public bool Show
            {
                get
                {
                    return Show;
                }
            }
        }

        public delegate void GOESShowChangeEventHandler(object sender, GOESShowChangeEventArgs e);

        public event GOESShowChangeEventHandler PrimaryGOESShowChanged; // the Primary GOES show checkbox has changed
        public event GOESShowChangeEventHandler SecondaryGOESShowChanged; // the Secondary GOES show checkbox has changed

        protected virtual void OnPrimaryGOESShowChanged(GOESShowChangeEventArgs e)
        {
            GOESShowChangeEventHandler handler = PrimaryGOESShowChanged;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }

        protected virtual void OnSecondaryGOESShowChanged(GOESShowChangeEventArgs e)
        {
            GOESShowChangeEventHandler handler = SecondaryGOESShowChanged;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }


        /// <summary>
        /// GOESDownloadRequested Event
        /// </summary>

        public event EventHandler GOESDownloadRequested;  // GOES download is requested

        protected virtual void OnGOESDownloadRequested(EventArgs e)
        {
            EventHandler handler = GOESDownloadRequested;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }
        #endregion

     
        

    }
}
